/*
 * cripple - command line CD ripper/encoder wrapper with cddb support
 *
 * Copyright (C) 2003-2005 Neil Phillips
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * tested and known to work on:
 *   - linux (2.4/2.6 at least)
 *   - win32-NT (4.0/5.0/5.1 at least)
 * untested support for:
 *   - *BSD
 *   - SunOS
 *   - OSF/1 (DEC etc.)
 * no support for:
 *   - IRIX
 *   - win32 95/98/Me
 *
 * win32-NT support is fully native, but the rest of cripple requires cygwin.
 * 
 * IRIX doesn't have a generic kernel CDROM interface like other systems, but
 * uses a set of libraries to access SCSI CDROMs (-lcdaudio -lmediad -lds). I
 * haven't implemented this yet (TODO).
 */

#include "cripple.h"

#if HAVE_W32API_WINDOWS_H
#  include <w32api/windows.h>
#else
#  include <sys/ioctl.h>
#endif

#if HAVE_LINUX_CDROM_H
#  include <linux/cdrom.h>
#endif
#if HAVE_SYS_CDIO_H
#  include <sys/cdio.h>		/* *BSD, solaris */
#endif
#ifdef HAVE_SYS_CDRIO_H
#  include <sys/cdrio.h>	/* *BSD -- only for CDIOREADSPEED define */
#endif
#if HAVE_IO_CAM_CDROM_H
#  include <io/cam/cdrom.h>	/* DEC */
#endif
#if HAVE_W32API_DDK_NTDDCDRM_H
#  include <w32api/ddk/ntddcdrm.h>
#endif

/*
 * just to be safe.
 */
#ifndef MAX_TRACKS
#  define MAX_TRACKS 100
#endif
#ifndef CDROM_LEADOUT
#  define CDROM_LEADOUT 0xaa
#endif

/*
 * in general ioctls on linux start with "CDROM", on *BSD "CDIO"/"CDRIO", on
 * DEC "CDROM_" and on win32 "IOCTL_CDROM"/"IOCTL_STORAGE".
 */


#ifdef __win32__

#include <ctype.h>

/*
 * drive letter here will get modified when a CD drive is found
 */
char win_cd_dev[] = "\\\\.\\D:";

/*
 * DeviceIoControl() needs a pointer to save length of returned data to
 */
static DWORD win_dummy;

static __inline__ void win_perror(const char *str)
{
	char *err_msg;

	FormatMessage(FORMAT_MESSAGE_ALLOCATE_BUFFER |
			FORMAT_MESSAGE_FROM_SYSTEM, NULL, GetLastError(),
			MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),
			(LPSTR)&err_msg, 0, NULL);
	fprintf(stderr, "%s: %s", str, err_msg);
}	

/*
 * this system specific mess gets folded into read_cdtoc_from_drive() below.
 */
static __inline__ int get_toc(HANDLE dev)
{
	CDROM_TOC toc;
	int i;

	if(!DeviceIoControl(dev, IOCTL_CDROM_READ_TOC, NULL, 0, &toc,
				sizeof(toc), &win_dummy, NULL)) {
		win_perror("DeviceIoControl(IOCTL_CDROM_READ_TOC)");
		fprintf(stderr, "is %s a CD drive with a valid CD?\n", cd_dev);
		return 1;
	}
	cd.tracks = toc.LastTrack;

	for(i=0; i <= cd.tracks; i++) {
		cd.cdtoc[i].min = toc.TrackData[i].Address[1];
		cd.cdtoc[i].sec = toc.TrackData[i].Address[2];
		cd.cdtoc[i].frame = toc.TrackData[i].Address[3]
			+ 75*(cd.cdtoc[i].min*60 + cd.cdtoc[i].sec);
		cd.cdtoc[i].control = toc.TrackData[i].Control;

		DEBUG_PRINTF("toc[%d].TrackNumber=0X%02hhx, .Control=0x%02hhx\n"
				, i, toc.TrackData[i].TrackNumber,
				toc.TrackData[i].Control);
	}
	return 0;
}

#else	/* not __win_32__ */

static __inline__ int get_toc(int dev)
{
#if defined(CDROMREADTOCHDR) && defined(CDROMREADTOCENTRY)
	struct cdrom_tochdr tochdr;
	struct cdrom_tocentry tocentry;
	int i;
	
	if(ioctl(dev, CDROMREADTOCHDR, &tochdr)) {
		if(ioctl(dev, CDROMREADTOCHDR, &tochdr)) { /* retry once */
			perror("ioctl(CDROMREADTOCHDR)");
			fprintf(stderr, "is %s a CD drive with a valid CD?\n",
					cd_dev);
			return 1;
		}
	}
	cd.tracks = tochdr.cdth_trk1;

	tocentry.cdte_format = CDROM_MSF;
	for(i=0; i <= cd.tracks; i++) {
		tocentry.cdte_track = (i < cd.tracks) ? i+1 : CDROM_LEADOUT;
		if(ioctl(dev, CDROMREADTOCENTRY, &tocentry)) {
			perror("ioctl(CDROMREADTOCENTRY)");
			return 1;
		}
		cd.cdtoc[i].min = tocentry.cdte_addr.msf.minute;
		cd.cdtoc[i].sec = tocentry.cdte_addr.msf.second;
		cd.cdtoc[i].frame = tocentry.cdte_addr.msf.frame
			+ 75*(cd.cdtoc[i].min*60 + cd.cdtoc[i].sec);
		cd.cdtoc[i].control = tocentry.cdte_ctrl;

		DEBUG_PRINTF("cdte_track=0x%02hhx cdte_addr=0x%02hhx "
				"cdte_ctrl=0x%02hhx cdte_datamode=0x%02hhx\n",
				tocentry.cdte_track, tocentry.cdte_adr,
				tocentry.cdte_ctrl, tocentry.cdte_datamode);
	}
#elif defined(CDIOREADTOCHEADER) && defined(CDIOREADTOCENTRYS)
	struct ioc_toc_header tochdr;
	struct cd_toc_entry entries[MAX_TRACKS];
	struct ioc_read_toc_entry rte;
	int i;

	if(ioctl(dev, CDIOREADTOCHEADER, &tochdr)) {
		if(ioctl(dev, CDIOREADTOCHEADER, &tochdr)) {
			perror("ioctl(CDIOREADTOCHEADER)");
			fprintf(stderr, "is %s a CD drive with a valid CD?\n",
					cd_dev);
			return 1;
		}
	}
	cd.tracks = tochdr.ending_track;
	
	rte.address_format = CD_MSF_FORMAT;
	rte.starting_track = 0;
	rte.data_len = sizeof(entries);
	rte.data = entries;
	if(ioctl(dev, CDIOREADTOCENTRYS, &rte) < 0) {
		perror("ioctl(CDIOREADTOCENTRYS)");
		return 1;
	}
	for(i=0; i <= cd.tracks; i++) {
		cd.cdtoc[i].min = entries[i].addr.msf.minute;
		cd.cdtoc[i].sec = entries[i].addr.msf.second;
		cd.cdtoc[i].frame = entries[i].addr.msf.frame
			+ 75*(cd.cdtoc[i].min*60 + cd.cdtoc[i].sec);
		cd.cdtoc[i].control = entries[i].control;

		DEBUG_PRINTF("entries[%d].track=0x%02hhx .control=0x%02hhx\n",
				i, entries[i].track, entries[i].control);
	}
#elif defined(CDROM_TOC_HEADER) && defined(CDROM_TOC_ENTRYS)
	struct cd_toc_header tochdr;
	struct cd_toc_entry entries[MAX_TRACKS];
	struct cd_toc rte;
	int i;
	
	if(ioctl(dev, CDROM_TOC_HEADER, &tochdr)) {
		if(ioctl(dev, CDROM_TOC_HEADER, &tochdr)) {
			perror("ioctl(CDROMREADTOCHDR)");
			fprintf(stderr, "is %s a CD drive with a valid CD?\n",
					cd_dev);
			return 1;
		}
	}
	cd.tracks = tochdr.th_ending_track;
	
	rte.toc_address_format = CDROM_MSF_FORMAT;
	rte.toc_starting_track = 0;
	rte.toc_alloc_length = sizeof(entries);
	rte.toc_buffer = (void *)entries;
	if(ioctl(dev, CDROM_TOC_ENTRYS, &rte) < 0) {
		perror("ioctl(CDROM_TOC_ENTRYS)");
		return 1;
	}
	for(i=0; i <= cd.tracks; i++) {
		cd.cdtoc[i].min = entries[i].te_absaddr.msf.m_units;
		cd.cdtoc[i].sec = entries[i].te_absaddr.msf.s_units;
		cd.cdtoc[i].frame = entries[i].te_absaddr.msf.f_units
			+ 75*(cd.cdtoc[i].min*60*75 + cd.cdtoc[i].sec);
		cd.cdtoc[i].control = entries[i].te_control;

		DEBUG_PRINTF("entries[%d].track=0x%02hhx .control=0x%02hhx\n",
			i, entries[i].te_track_number, entries[i].te_control);
	}
#endif
	return 0;
}	

#endif /* not __win32__ */


int read_cdtoc_from_drive(void)
{
#ifdef __win32__
	HANDLE dev;
	char devstr[] = "D:\\";

	if(!*cd_dev) {
		while(GetDriveType((LPCTSTR)devstr) != DRIVE_CDROM
				&& ++(*devstr) <= 'Z');
		if(*devstr > 'Z') {
			fputs("can't find a cdrom drive!\n", stderr);
			return 1;
		}
		win_cd_dev[4] = *devstr;
		cd_dev = win_cd_dev;
	} else if(isalpha(*cd_dev) && cd_dev[1] == ':' && cd_dev[2] == '\0') {
		win_cd_dev[4] = *cd_dev;
		cd_dev = win_cd_dev;
	}

	dev = CreateFile(cd_dev, GENERIC_READ, FILE_SHARE_READ
			| FILE_SHARE_WRITE, NULL, OPEN_EXISTING,
			FILE_ATTRIBUTE_NORMAL, NULL);
	if(dev == INVALID_HANDLE_VALUE) {
		win_perror("CreateFile()");
		fprintf(stderr, "error opening: %s\n", cd_dev);
		return 1;
	}
#else	/* not win32 */
	int dev;
	
	if((dev = open(cd_dev, O_RDONLY | O_NONBLOCK)) == -1)
		dev = check_open(cd_dev, O_RDONLY | O_NONBLOCK, 0);
#endif


#ifdef CDROMCLOSETRAY
	ioctl(dev, CDROMCLOSETRAY);
#elif defined(CDIOCLOSE)
	ioctl(dev, CDIOCLOSE);
#elif defined(IOCTL_STORAGE_LOAD_MEDIA)
	DeviceIoControl(dev, IOCTL_STORAGE_LOAD_MEDIA, NULL, 0, NULL, 0,
			&win_dummy, NULL);
#endif

	
#ifdef CDROM_SELECT_SPEED
	if(ioctl(dev, CDROM_SELECT_SPEED, drive_speed))
		perror("ioctl(CDROM_SELECT_SPEED)");
#elif defined(CDRIOREADSPEED)
	if(ioctl(dev, CDRIOREADSPEED, drive_speed))
		perror("ioctl(CDRIOREADSPEED)");
#endif

	/*
	 * system specific stuff to fill out cd.cdtoc[] and cd.tracks
	 */
	if(get_toc(dev)) {
#ifdef __win32__
		CloseHandle(dev);
#else	
		close(dev);
#endif
		return 1;
	}

	cd.tot_secs = ((cd.cdtoc[cd.tracks].min*60) + cd.cdtoc[cd.tracks].sec);
	cd.play_secs = cd.tot_secs - (cd.cdtoc[0].min * 60) - cd.cdtoc[0].sec;

#ifdef __win32__
	return !CloseHandle(dev);
#else
	return close(dev);
#endif
}
	



void cddb_discid(void)
{
	unsigned long i, n, secs;
	
	for(i = n = 0; i < cd.tracks; i++) {
		secs = (cd.cdtoc[i].min * 60) + cd.cdtoc[i].sec;
		for(; secs; secs /= 10)
			n += (secs % 10);	/* sum of digits */
	}

	cd.discid = (n % 0xff) << 24 | cd.play_secs << 8 | cd.tracks;
}



int eject_cdrom(void)
{
#if defined(CDROMEJECT) || defined(CDIOEJECT) || defined(CDROM_EJECT_CADDY)
	int dev = open(cd_dev, O_RDONLY | O_NONBLOCK);
	if(!dev) return 1;
#  ifdef CDROMEJECT
	if(ioctl(dev, CDROMEJECT)) return 1;
#  elif defined(CDIOEJECT)
	if(ioctl(dev, CDIOEJECT)) return 1;
#  elif defined(CDROM_EJECT_CADDY)
	if(ioctl(dev, CDROM_EJECT_CADDY)) return 1;
#  endif
	return close(dev);
#elif defined(IOCTL_STORAGE_EJECT_MEDIA)
	HANDLE dev = CreateFile(cd_dev, GENERIC_READ, FILE_SHARE_READ
			| FILE_SHARE_WRITE, NULL, OPEN_EXISTING,
			FILE_ATTRIBUTE_NORMAL, NULL);
	if(dev == INVALID_HANDLE_VALUE) return 1;
	if(!DeviceIoControl(dev, IOCTL_STORAGE_EJECT_MEDIA, NULL, 0, NULL, 0,
			&win_dummy, NULL)) return 1;
	return !CloseHandle(dev);
#else
	return 0;
#endif
}



void print_cdtoc()
{
	int i;
	const char data_track[] = "data", audio_track[] = "audio";

	fprintf(stderr, "cdtoc[] contents:\n"
			"-----------------");
	for(i=0; i <= cd.tracks; i++) {
		fprintf(stderr, "\ntrack: %3d   min: %2d    sec: %2d    "
				"frame: %6d    content: %s", 
			(i<cd.tracks) ? i : CDROM_LEADOUT,
			cd.cdtoc[i].min, cd.cdtoc[i].sec, cd.cdtoc[i].frame,
			cd.cdtoc[i].control & CONTROL_DATA ?
						data_track : audio_track);
		if(cd.cdtoc[i].control & CONTROL_PREEMPH)
			fputs(", preemph", stderr);
		if(cd.cdtoc[i].control & CONTROL_COPY)
			fputs(", copy", stderr);
		if(cd.cdtoc[i].control & CONTROL_4CHAN)
			fputs(", 4channel", stderr);
	}
	fprintf(stderr, "\n\nno. tracks:\t%d\n"
			"discid:\t\t%08lx\n\n", cd.tracks, cd.discid);
}


#define buf_msf ((struct cdrom_msf *)(&buf))
int read_mode2_track(int trk, int fd)
{
#ifdef CDROMREADMODE2
	int dev, frame = cd.cdtoc[trk].frame, i;
	char buf[2336];

	if((dev = open(cd_dev, O_RDONLY)) == -1)
		dev = check_open(cd_dev, O_RDONLY, 0);

	for(; frame < cd.cdtoc[trk+1].frame; frame++) {
		buf_msf->cdmsf_min0 = buf_msf->cdmsf_min1 = frame / 75 / 60;
		buf_msf->cdmsf_sec0 = buf_msf->cdmsf_sec1 = frame / 75 % 60;
		buf_msf->cdmsf_frame0 = buf_msf->cdmsf_frame1 = frame % 75;

		for(i=0; ioctl(dev, CDROMREADMODE2, buf) && i<=4; i++) {
			fprintf(stderr, "frame: %d, retry %d: ", frame, i);
			perror("ioctl(CDROMREADMODE2)");
		}
		if(i>4) {
			fprintf(stderr, "unable to read frame %d, writing zeros...\n", frame);
			memset(buf, 0, 2336);
		}
/*		check_write(fd, buf+8, 2048);*/
		check_write(fd, buf, 2336);
			
	}
#endif
	return 0;
}

