/*
 * cripple - command line CD ripper/encoder wrapper with cddb support
 *
 * Copyright (C) 2003/2004 Neil Phillips
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * add an ID3v1.1 tag to a file
 *
 * this will append a tag to any file - regarless of if it alredy has a tag.
 */

#include "cripple.h"

#if HAVE_STRINGS_H
#  include <strings.h>
#endif

struct id3_tag {
	char tag_pad[3];
	char title[30];
	char artist[30];
	char album[30];
	char year[4];
	char comment[28];
	char comment_end;
	char track;
	char genre;
} __attribute__((packed));



/* information here is taken from:
 * 
 * libid3tag - ID3 tag manipulation library
 * Copyright (C) 2000-2001 Robert Leslie
 */
const char *id3genres[] = {
/* 0 */		"Blues",
		"Classic Rock",
		"Country",
		"Dance",
		"Disco",
		"Funk",
		"Grunge",
		"Hip-Hop",
		"Jazz",
		"Metal",
/* 10 */	"New Age",
		"Oldies",
		"Other",
		"Pop",
		"R&B",
		"Rap",
		"Reggae",
		"Rock",
		"Techno",
		"Industrial",
/* 20 */	"Alternative",
		"Ska",
		"Death Metal",
		"Pranks",
		"Soundtrack",
		"Euro-Techno",
		"Ambient",
		"Trip-Hop",
		"Vocal",
		"Jazz+Funk",
/* 30 */	"Fusion",
		"Trance",
		"Classical",
		"Instrumental",
		"Acid",
		"House",
		"Game",
		"Sound Clip",
		"Gospel",
		"Noise",
/* 40 */	"AlternRock",
		"Bass",
		"Soul",
		"Punk",
		"Space",
		"Meditative",
		"Instrumental Pop",
		"Instrumental Rock",
		"Ethnic",
		"Gothic",
/* 50 */	"Darkwave",
		"Techno-Industrial",
		"Electronic",
		"Pop-Folk",
		"Eurodance",
		"Dream",
		"Southern Rock",
		"Comedy",
		"Cult",
		"Gangsta",
/* 60 */	"Top 40",
		"Christian Rap",
		"Pop/Funk",
		"Jungle",
		"Native American",
		"Cabaretn",
		"New Wave",
		"Psychedelic",
		"Rave",
		"Showtunes",
/* 70 */	"Trailer",
		"Lo-Fi",
		"Tribal",
		"Acid Punk",
		"Acid Jazz",
		"Polka",
		"Retron",
		"Musical",
		"Rock & Roll",
		"Hard Rock",
/* 80 */	"Folk",
		"Folk/Rock",
		"National Folk",
		"Swing",
		"Fast-Fusion",
		"Bebob",
		"Latin",
		"Revival",
		"Celtic",
		"Bluegrass",
/* 90 */	"Avantgarde",
		"Gothic Rock",
		"Progressive Rock",
		"Psychedelic Rock",
		"Symphonic Rock",
		"Slow Rock",
		"Big Band",
		"Chorus",
		"Easy Listening",
		"Acoustic",
/* 100 */	"Humour",
		"Speech",
		"Chanson",
		"Opera",
		"Chamber Music",
		"Sonata",
		"Symphony",
		"Booty Bass",
		"Primus",
		"Porn Groove",
/* 110 */	"Satire",
		"Slow Jam",
		"Club",
		"Tango",
		"Samba",
		"Folklore",
		"Ballad",
		"Power Ballad",
		"Rhythmic Soul",
		"Freestyle",
/* 120 */	"Duet",
		"Punk Rock",
		"Drum Solo",
		"A Cappella",
		"Euro-House",
		"Dance Hall",
		"Goa",
		"Drum & Bass",
		"Club-House",
		"Hardcore",
/* 130 */	"Terror",
		"Indie",
		"BritPop",
		"Negerpunk",
		"Polsk Punk",
		"Beat",
		"Christian Gangsta Rap",
		"Heavy Metal",
		"Black Metal",
		"Crossover",
/* 140 */	"Contemporary Christian",
		"Christian Rock",
		"Merengue",
		"Salsa",
		"Thrash Metal",
		"Anime",
		"JPop",
		"Synthpop",
		NULL};

static __inline__ int id3_genre(const char *str)
{
	int i;

	for(i=0; id3genres[i] && strcasecmp(str, id3genres[i]); i++);
	return id3genres[i] ? i : -1;
}


void write_id3tag(const char *filename, int track)
{
	struct id3_tag tag = {{'T', 'A', 'G'}};
	int fd;
	const char *str;

	if(cd.artists[track])
		strncpy(tag.artist, cd.artists[track], 30);
	else if(cd.artist)
		strncpy(tag.artist, cd.artist, 30);

	if(cd.album)
		strncpy(tag.album, cd.album, 30);
	if(cd.track_names[track])
		strncpy(tag.title, cd.track_names[track], 30);
	if(cd.comment)
		strncpy(tag.comment, cd.comment, 28);
	if(cd.year)
		strncpy(tag.year, cd.year, 4);

	tag.track = track+1;

	if(cd.genre) {
		if(!(tag.genre = atoi(cd.genre)) ) {
		    /* fuzzy genre matching -- try skipping first words e.g.
		     * 'Industrial Metal'     --> 'Metal'
		     * 'Satanic Thrash Metal' --> 'Thrash Metal' */
		    for(str = cd.genre; (tag.genre = id3_genre(str)) == -1 &&
				    	(str = strchr(str, ' ')); str++);
		    if(tag.genre == -1)
			fprintf(stderr, "WARNING: '%s' is not a recognised ID3"
					" genre\n", cd.genre);
		}
	} else tag.genre = -1;


	DEBUG_PRINTF("Artist:\t\t%.30s\n"
			"Album:\t\t%.30s\n"
			"Title:\t\t%.30s\n"
			"Comment:\t%.28s\n"
			"Year:\t\t%.4s\n"
			"Track:\t\t%hhd\n"
			"Genre no.:\t%hhd\n",
			tag.artist, tag.album, tag.title, tag.comment,
			tag.year, tag.track, tag.genre);

	fd = check_open(filename, O_WRONLY|O_APPEND, 0);
	check_write(fd, &tag, sizeof(struct id3_tag) /* 128 */ );
	fprintf(stderr, "\nID3 tag successfully written\n\n");

	close(fd);
}
