 /*
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Initial Developer of the Original Code is
 * anonymous
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
*/

#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <ctype.h>
#include <string.h>

struct ignore_node {
  char *str;
  struct ignore_node *next;
};

struct ignore_node *ignore_list = 0;
struct ignore_node *ignore_list_tail = 0;

/*0=false, 1=true*/
int line_contains_ignore_pattern(const char *line)
{
  struct ignore_node *node;
  for (node=ignore_list; node; node=node->next) {
    if (0!=strstr(line, node->str)) {
      /*found*/
      return 1;
    }
  }
  return 0;
}

int main(int argc, char *argv[])
{
  const int BUFMAX=16384-10;
  char line[BUFMAX+10];
  int i, c, is_cr_line;
  time_t last_print_time = 0;
  int seconds_skip = -1;
  FILE *cr_line_destination = stdout;
  FILE *output_destination = stdout;
  int bad_usage = 0;
  int prev_line_was_cr_swallowed=0;

  if (argc < 2) {
    bad_usage = 1;
  }
  else {
    for (i=2; i<=argc; ++i) {
      if (isdigit(argv[i-1][0])) {
        if (seconds_skip != -1) {
          bad_usage=1;
          break;
        }
        seconds_skip = atoi(argv[i-1]);
      }
      else if (0==strcmp(argv[i-1], "-out-to-stderr")) {
        output_destination=stderr;
      }
      else if (0==strcmp(argv[i-1], "-cr-to-stderr")) {
        cr_line_destination=stderr;
      }
      else if (0==strcmp(argv[i-1],"-ignore")) {
        if (i==argc) {
          /*need more arguments*/
          bad_usage=1;
          break;
        }
        else {
          char *ignore_this;
          struct ignore_node *new_node;

          ++i;
          new_node = malloc(sizeof(struct ignore_node));
          if (!new_node) {
            abort();
          }
          new_node->next = 0;
          new_node->str = strdup(argv[i-1]);
          if (!new_node->str) {
            abort();
          }
          if (!ignore_list) {
            /*first node*/
            ignore_list = new_node;
            ignore_list_tail = new_node;
          }
          else {
            ignore_list_tail->next = new_node;
            ignore_list_tail = new_node;
          }
        }
      }
      else {
        bad_usage=1;
        break;
      }
    }
  }
  
  if (seconds_skip == -1) {
    seconds_skip = 1; /*sane default*/
  }
  
  if (bad_usage) {
    puts("crswallow\n"
      "usage: crswallow <seconds to swallow> [-out-to-stderr] [-cr-to-stderr] [-ignore <STRING>] ...\n"
      "swallow excessize console output separated by carriage return\n"
      "consecutive output separated by cr will be filtered to one segment per requested period\n"
      "default: 1 second"
      "special value 0 (zero) means: swallow lines containing cr completely\n"
      "by default, all input is forwarded to stdout\n"
      "use -cr-to-stderr to forward all lines containing cr to stderr\n"
      "use -out-to-stderr to forward all lines NOT containing cr to stderr\n"
    );
    return 0;
  }
  
  is_cr_line=0;
  i=0;
  while (1) {
    if (i >= BUFMAX) {
      /*overflow, can't filter, flush and continue*/
      line[i] = 0;
      fputs(line, output_destination);
      i=0;
      is_cr_line=0;
      continue;
    }
    c = fgetc(stdin);
    if (c == 0) {
      return -1;
    }
    if (c == EOF) {
      line[i] = 0;
      if (seconds_skip!=0 || !is_cr_line) {
        fputs(line, (is_cr_line?cr_line_destination:output_destination));
      }
      return 0;
    }
    if (c == '\n') {
      line[i] = c;
      ++i;
      if (i==1) {
        /*keep and include (or swallow) later*/
      }
      else if (line_contains_ignore_pattern(line)) {
        /*swallow line*/
        i=0;
      }
      else {
        line[i] = 0;
        if (seconds_skip!=0 || !is_cr_line) {
          fputs(line, (is_cr_line?cr_line_destination:output_destination));
        }
        i=0;
        last_print_time = 0;
        is_cr_line=0;
      }
      continue;
    }
    if (c == '\r') {
      is_cr_line=1;
      time_t now = time(0);
      if (!last_print_time || now >= last_print_time + seconds_skip) {
        /* print */
        line[i] = c;
        ++i;
        line[i] = 0;
        if (seconds_skip!=0) {
          fputs(line, cr_line_destination);
          fflush(cr_line_destination);
        }
        i=0;
        last_print_time = now;
      }
      else {
        /* discard */
        i=0;
      }
      continue;
    }
    line[i] = c;
    ++i;
  };
}
