#!/bin/bash

############################################################################
#    Copyright (C) 2008-2009 by Jaroslaw Zachwieja <grok@zyxxyz.eu>        #
#                                                                          #
#    This program is free software; you can redistribute it and/or modify  #
#    it under the terms of the GNU General Public License as published by  #
#    the Free Software Foundation; either version 3 of the License, or     #
#    (at your option) any later version.                                   #
#                                                                          #
#    This program is distributed in the hope that it will be useful,       #
#    but WITHOUT ANY WARRANTY; without even the implied warranty of        #
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         #
#    GNU General Public License for more details.                          #
#                                                                          #
#    You should have received a copy of the GNU General Public License     #
#    along with this program.  If not, see <http://www.gnu.org/licenses/>. #
############################################################################

# Dependencies:
# 1) mplayer/mencoder that supports x264 encoding
# 2) mkvtoolnix (provides mkvmerge)
# 3) lsdvd
# 4) dvdxchap

# These are defaults, edit autoripper.conf instead!
DEFAULT_TITLES="LONGEST"
DEFAULT_SHORTEST_FIRST="0"
DEFAULT_OUTPUT="/tmp/ISO"
DEFAULT_FINAL_DESTINATION="/done"
DEFAULT_UPSAMPLE="false"
DEFAULT_DEINT=""
DEFAULT_SAMPLE_POINTS=20
DEFAULT_QUALITY="HIGH"
DEFAULT_FORCE_PULLUP="1"
 
# Load configuration
if (test -s ./autoripper.conf)
then
	. ./autoripper.conf
	echo "Found config file './autoripper.conf', loading."
elif (test -s ~/.autoripper.conf)
then
	. ~/.autoripper.conf
	echo "Found config file '~/.autoripper.conf', loading."
elif (test -s /etc/autoripper.conf)
then
	. /etc/autoripper.conf
	echo "Found config file '/etc/autoripper.conf', loading."
else
	echo "No config file found, using defaults."
fi

if (test -z $TITLES)
then
    TITLES="$DEFAULT_TITLES"
fi

if (test -z $SHORTEST_FIRST)
then
    SHORTEST_FIRST=$DEFAULT_SHORTEST_FIRST
fi

if (test -z $OUTPUT)
then
    OUTPUT="$DEFAULT_OUTPUT"
fi

if (test -z $FINAL_DESTINATION)
then
    FINAL_DESTINATION="$DEFAULT_FINAL_DESTINATION"
fi

if (test -z $UPSAMPLE)
then
    UPSAMPLE="$DEFAULT_UPSAMPLE"
fi

if (test -z $SAMPLE_POINTS)
then
    SAMPLE_POINTS="$DEFAULT_SAMPLE_POINTS"
fi

if (test -z $FORCE_PULLUP)
then
    FORCE_PULLUP="$DEFAULT_FORCE_PULLUP"
fi

if (test -z $QUALITY)
then
    QUALITY="$DEFAULT_QUALITY"
fi

if [ "$UPSAMPLE" == "false" ]
then
    case $QUALITY in
        "HIGH") CRF="18"
                ;;
        "MEDIUM") CRF="20"
                ;;
        "LOW") CRF="22"
                ;;
    esac
else
    case $QUALITY in
        "HIGH") CRF="20"
                ;;
        "MEDIUM") CRF="22"
                ;;
        "LOW") CRF="24"
                ;;
    esac
fi

if (test -z $DEINT)
then
    DEINT="$DEFAULT_DEINT"
fi

if !(test -d ${OUTPUT})
then
	echo "OUTPUT $OUTPUT does not exist or is not a directory"
	exit 1
fi

if !(test -d ${FINAL_DESTINATION})
then
	echo "FINAL_DESTINATION $FINAL_DESTINATION does not exist or is not a directory"
	exit 1
fi


INPUT="$1"

if [ "$INPUT" == "" ] || [ "$INPUT" == "-h" ] || [ "$INPUT" == "--help" ]
then
        echo "Usage: $0 <iso image|dvd device> [output name prefix]"
        echo "image rip is much faster than device rip."
        exit 0
fi

if !(mencoder -ovc help | grep -i x264)
then
    echo "This version of mencoder does not support x264."
    echo "Try installing restricted mplayer packages and libraries for your distribution."
    exit 1
else
    echo "mencoder seems to have x264 support."
fi


# Unlock the drive if we're ripping from a disk instead of an image
# ... and fetch the DISC_ID
DISC_ID="`mplayer dvd:// -dvd-device ${INPUT} -ss 1 -endpos 1 -identify -really-quiet -vo null -ao null | grep ID_DVD_DISC_ID | sed 's#ID_DVD_DISC_ID=##' | sed 's# ##g'`"

if [ -n "$2" ]
then
    TITLE_PREFIX="$2"
else
    TITLE_PREFIX="`lsdvd ${INPUT} | grep 'Disc Title:' | cut -d ' ' -f 3`"
    if [ "$TITLE_PREFIX" == "unknown" ] || [ "$TITLE_PREFIX" == "UNKNOWN" ] || [ "$TITLE_PREFIX" == "Unknown" ]
    then
        echo "Disc title unknown, using DISC_${DISC_ID}"
        TITLE_PREFIX="DISC_${DISC_ID}"
    fi
fi

# Clean up any aborted transcodes in temporary directory.
rm -f ${OUTPUT}/${TITLE_PREFIX}_??.*

# the _DECIMAL variables are required because bash interprets '0'-prefixed numbers as octal.
# Sometimes we need that as string (grepping), sometimes as number (counting).

if [ "$TITLES" == "LONGEST" ]
then
    ALL_TITLES="`lsdvd ${INPUT} | grep 'Length:' | sort -rk 4 |head -1 |cut -d ' ' -f 2|cut -d ',' -f 1`"
elif [ "$TITLES" == "ALL" ]
then
    SORT_R="-r"
    if [ "$SHORTEST_FIRST" == "1" ]; then SORT_R=""; fi
    ALL_TITLES=`lsdvd ${INPUT} | grep 'Length:' | grep -v 'Length: 00:00:' | sort ${SORT_R} -k 4 |cut -d ' ' -f 2|cut -d ',' -f 1`
else
    ALL_TITLES="$TITLES"
fi

echo "Encoding $TITLE_PREFIX, Titles:"
echo $ALL_TITLES

# Test for filename conflicts
for A_TITLE in $ALL_TITLES;
do
    TITLE=${TITLE_PREFIX}_${A_TITLE}

    if [ -a ${FINAL_DESTINATION}/${TITLE}.mkv ]
    then
        echo "destination file ${FINAL_DESTINATION}/${TITLE}.mkv already exists, exiting"
        exit 1
    fi
done

# Loop the desired titles
for A_TITLE in $ALL_TITLES;
do
    TITLE=${TITLE_PREFIX}_${A_TITLE}
    
    A_TITLE_DECIMAL="`echo $A_TITLE | sed 's/0*\([0-9]\)/\1/'`"
    
    AUDIO_STREAMS_NUMBER=
    AUDIO_STREAMS_NUMBER_DECIMAL=
    AUDIO_STREAMS_IDS=
    AUDIO_STREAMS=
    SUBTITLE_STREAMS_NUMBER=
    SUBTITLE_STREAMS_NUMBER_DECIMAL=
    SUBTITLE_STREAMS_IDS=
    SUBTITLE_STREAMS=
    
    AUDIO_STREAMS_NUMBER="`lsdvd -t ${A_TITLE} ${INPUT}  | grep "^Title:" | cut -d " " -f 11|cut -d "," -f 1`"
    AUDIO_STREAMS_NUMBER_DECIMAL="`echo $AUDIO_STREAMS_NUMBER | sed 's/0*\([0-9]\)/\1/'`"
    
    AUDIO_STREAMS_IDS=`lsdvd -a -t ${A_TITLE} ${INPUT} | grep -C ${AUDIO_STREAMS_NUMBER_DECIMAL} "Title: ${A_TITLE}" | tail -n ${AUDIO_STREAMS_NUMBER_DECIMAL} | cut -d ':' -f 10|sed -e 's/\ //g'`
    AUDIO_STREAMS="`for N in $AUDIO_STREAMS_IDS ; do AUDIO_STREAMS="$AUDIO_STREAMS $N" ; done ; echo $AUDIO_STREAMS`"
    
    SUBTITLE_STREAMS_NUMBER="`lsdvd -t ${A_TITLE} ${INPUT} | grep "^Title:" | cut -d " " -f 13|cut -d "," -f 1`"
    SUBTITLE_STREAMS_NUMBER_DECIMAL="`echo $SUBTITLE_STREAMS_NUMBER | sed 's/0*\([0-9]\)/\1/'`"
    
    SUBTITLE_STREAMS_IDS=`lsdvd -s -t ${A_TITLE} ${INPUT} | grep -C ${SUBTITLE_STREAMS_NUMBER_DECIMAL} "Title: ${A_TITLE}" | tail -n ${SUBTITLE_STREAMS_NUMBER_DECIMAL} | grep -v "xx - Unknown" | cut -d ':' -f 5 | cut -d "," -f 1 | sed -e 's/\ //g'`
    SUBTITLE_STREAMS="`for N in $SUBTITLE_STREAMS_IDS ; do SUBTITLE_STREAMS="$SUBTITLE_STREAMS $N" ; done ; echo $SUBTITLE_STREAMS`"
    
    echo "Title: $A_TITLE_DECIMAL, audio streams: $AUDIO_STREAMS, subtitle streams: $SUBTITLE_STREAMS"
    
    dvdxchap -t ${A_TITLE_DECIMAL} ${INPUT}  | sed -e "s/Chapter\ //g" > ${OUTPUT}/${TITLE}.chapters

    echo "Calculating default frame duration (ms)."
    FPS="`mplayer -dvd-device $INPUT dvd://${A_TITLE_DECIMAL} -identify -ao null -vo null -ss 1 -endpos 1 2>/dev/null |grep ID_VIDEO_FPS |cut -d '=' -f 2`"
    
    case $FPS in
    
        23.976) OFPS="24000/1001"
		DEFD="41.708ms"
		;;
        24.000) OFPS="24"
		DEFD="41.666ms"
		;;
        25.000) OFPS=$FPS
		DEFD="40.000ms"
		;;
        29.970) if [ "$FORCE_PULLUP" == "1" ]
		then
			OFPS="24000/1001"
			DEFD="41.708ms"
		else
			OFPS="30000/1001"
			DEFD="33.367ms"
		fi
            ;;
        30.000) if [ "$FORCE_PULLUP" == "1" ]
		then
			OFPS=$FPS
			DEFD="33.333ms"
		else
			OFPS="24"
			DEFD="41.666ms"
		fi
		;;
    esac
    
    echo "Frame rate: ${OFPS}, frame duration: $DEFD"
    
    COUNTER=0
    rm -f $OUTPUT/${TITLE}.smap
    if [ "$SUBTITLE_STREAMS_NUMBER_DECIMAL" != "0" ]
    then
        for SID in $SUBTITLE_STREAMS
        do
            echo "Extracting subtitle $SID"
            SKIP=0
            SUBTITLE_LANGUAGE_CODE="`lsdvd -t ${A_TITLE_DECIMAL} -s $INPUT | grep $SID | cut -d ':' -f 3 | cut -d ' ' -f 2`"
            SUBTITLE_LANGUAGE_NAME="`lsdvd -t ${A_TITLE_DECIMAL} -s $INPUT | grep $SID | cut -d ':' -f 3 | cut -d ' ' -f 4 | cut -d ',' -f 1`"

            RUNARGS="-nosound -ovc frameno -vf pullup,softskip,harddup -ofps ${OFPS} -vobsubout ${OUTPUT}/${TITLE} -o /dev/null -sid $SID -vobsuboutid $SUBTITLE_LANGUAGE_CODE -vobsuboutindex $COUNTER dvd://${A_TITLE_DECIMAL} -dvd-device ${INPUT}"
            echo "mencoder $RUNARGS"
            mencoder $RUNARGS 1>&2 || SKIP=1
    
            if [ "$SKIP" == "0" ]
            then
                echo "Writing entry for subtitle $SID into subtitle map ${OUTPUT}/${TITLE}.smap"
                echo "$SID $SUBTITLE_LANGUAGE_CODE $SUBTITLE_LANGUAGE_NAME index=$COUNTER" >> $OUTPUT/${TITLE}.smap
                COUNTER=$[${COUNTER}+1]
            else
                echo "Ripping subtitle $SID failed, skipping it."
            fi
        done
        cat $OUTPUT/${TITLE}.smap
    fi

    rm -f $OUTPUT/${TITLE}.amap
    COUNTER=0
    for AID in $AUDIO_STREAMS
    do
        echo "Extracting audio track $AID"
        SKIP=1
        AUDIO_LANGUAGE_CODE="`lsdvd -t ${A_TITLE_DECIMAL} -a $INPUT | grep $AID | cut -d ':' -f 3 | cut -d ' ' -f 2`"
        AUDIO_LANGUAGE_NAME="`lsdvd -t ${A_TITLE_DECIMAL} -a $INPUT | grep $AID | cut -d ':' -f 3 | cut -d ' ' -f 4 | cut -d ',' -f 1`"
    
        RUNARGS="-aid $AID -vo null -vc dummy -dumpaudio -dumpfile ${OUTPUT}/${TITLE}.$AID dvd://${A_TITLE_DECIMAL} -dvd-device ${INPUT}"
        echo "mplayer $RUNARGS"
        mplayer $RUNARGS 1>&2 || SKIP=1
    
        if [ -s "${OUTPUT}/${TITLE}.$AID" ]
        then
            SKIP=0
        fi
    
        if [ "$SKIP" == "0" ]
        then
            echo "Writing entry for audio $AID into audio map ${OUTPUT}/${TITLE}.amap"
            echo "$AID $AUDIO_LANGUAGE_CODE $AUDIO_LANGUAGE_NAME index=$COUNTER" >>$OUTPUT/${TITLE}.amap
            COUNTER=$[${COUNTER}+1]
        else
            echo "Ripping audio $AID failed, skipping it."
        fi
    done

    if [ "$COUNTER" == "0" ]
    then
        echo "dvd with bugs. attempting to extract a default audio track"
        RUNARGS="-vo null -vc dummy -dumpaudio -dumpfile ${OUTPUT}/${TITLE}.0x80 dvd://${A_TITLE_DECIMAL} -dvd-device ${INPUT}"
        echo "mplayer $RUNARGS"
        mplayer $RUNARGS 1>&2
        if [ -s "${OUTPUT}/${TITLE}.0x80" ]
        then
            echo "Writing entry for audio into audio map ${OUTPUT}/${TITLE}.amap"
            echo "0x80 xx Unknown index=0" >>$OUTPUT/${TITLE}.amap
        fi
    fi

    cat $OUTPUT/${TITLE}.amap
   
    echo "Ripping main video stream."
    
    TITLE_LENGTH="`lsdvd -t ${A_TITLE_DECIMAL} ${INPUT} | cut -d ':' -f 3-4|cut -d ' ' -f 2`"
    HOURS="`echo $TITLE_LENGTH|cut -d ':' -f 1|sed 's/0*\([0-9]\)/\1/'`"
    MINUTES="`echo $TITLE_LENGTH|cut -d ':' -f 2|sed 's/0*\([0-9]\)/\1/'`"
    TITLE_LENGTH_SECONDS="$[${HOURS}*3600 + ${MINUTES}*60]"
    
    echo "Title length: ${TITLE_LENGTH_SECONDS} seconds"
    SAMPLE_STEP=$[${TITLE_LENGTH_SECONDS}/${SAMPLE_POINTS}]
    
    Xc=0
    Yc=0
    Wc=1000
    Hc=1000
    COUNTER=0
    for SAMPLE in `seq 1 ${SAMPLE_STEP} ${TITLE_LENGTH_SECONDS}`
    do
    
        COUNTER=$[${COUNTER}+1]
        echo -n "Crop detect pass $COUNTER, position $SAMPLE seconds, 200 frames "
        CROP="`mplayer -dvd-device $INPUT dvd://${A_TITLE_DECIMAL} -vo null -nosound -vf cropdetect -ss $SAMPLE -frames 200 -fps 200 2>/dev/null |grep '[CROP]' |tail -n 1 |cut -d '(' -f 2 | cut -d ')' -f 1`"
        echo -n "$CROP "
        X="`echo $CROP | cut -d '=' -f 2 | cut -d ':' -f 1`"
        Y="`echo $CROP | cut -d '=' -f 2 | cut -d ':' -f 2`"
        W="`echo $CROP | cut -d '=' -f 2 | cut -d ':' -f 3`"
        H="`echo $CROP | cut -d '=' -f 2 | cut -d ':' -f 4`"
    
        if [ "$Xc" -lt "$X" ]
        then
            Xc=$X
            echo -n "X adjusted "
        fi
    
        if [ "$Yc" -lt "$Y" ]
        then
            Yc=$Y
            echo -n "Y adjusted "
        fi
    
        if [ "$Wc" -gt "$W" ]
    
        then
            Wc=$W
            echo -n "W adjusted "
        fi
    
        if [ "$Hc" -gt "$H" ]
        then
            Hc=$H
            echo -n "H adjusted "
        fi
        echo "Settled for $Xc $Yc $Wc $Hc."
    
    done
    
    CROP="crop=$Xc:$Yc:$Wc:$Hc"
    
    echo $CROP
    
    # WARNING: some versions of mplayer seem to ignore the "-noautosub" option and insist
    # on burning the subtitles into the stream anyway. If this happens, use "-sid 4096" instead,
    # it asks mplayer to use a subtitle stream ID that is well outside the usual range.
    
    if [ "$UPSAMPLE" == "false" ]
    then
        echo "Standard encode"
        RUNARGS="-nosound -noautosub dvd://${A_TITLE_DECIMAL} -dvd-device ${INPUT} -vf pullup,softskip,pp=${DEINT}ha/va/dr,${CROP},harddup -ovc x264 -x264encopts threads=auto:crf=${CRF}:level_idc=31:bframes=3:nodct_decimate:trellis=2:nob_pyramid:me=umh:mixed_refs:weight_b -ofps ${OFPS} -of rawvideo -o ${OUTPUT}/${TITLE}.x264"
        echo "mencoder $RUNARGS"
        ionice -c 3 nice -n 19 mencoder $RUNARGS 1>&2
    else
        echo "Upsampled encode"
        RUNARGS="-nosound -noautosub dvd://${A_TITLE_DECIMAL} -dvd-device ${INPUT} -vf pullup,softskip,pp=${DEINT}ha/va/dr,${CROP},scale=1280:-10,harddup -sws 9 -ovc x264 -x264encopts threads=auto:crf=${CRF}:level_idc=40:bframes=3:nodct_decimate:trellis=2:nob_pyramid:me=umh:mixed_refs:weight_b -ofps ${OFPS} -of rawvideo -o ${OUTPUT}/${TITLE}.x264"
        echo "mencoder $RUNARGS"
        ionice -c 3 nice -n 19 mencoder $RUNARGS 1>&2
    fi
    
    ALLAUDIO="--default-track 0"
    AVAILABLE_AUDIO="`cat ${OUTPUT}/${TITLE}.amap | cut -d ' ' -f 1`"
    
    for ATRACK in $AVAILABLE_AUDIO
    do
        AINDEX="`grep $ATRACK ${OUTPUT}/${TITLE}.amap | cut -d ' ' -f 4 | cut -d '=' -f 2`"
        ALANGUAGE_CODE="`grep $ATRACK ${OUTPUT}/${TITLE}.amap | cut -d ' ' -f 2`"
        ALANGUAGE_NAME1="`grep $ATRACK ${OUTPUT}/${TITLE}.amap | cut -d ' ' -f 3`"
        ALANGUAGE_NAME2="`mkvmerge --list-languages | grep -v code | cut -d '|' -f 2,3 | sed -e 's/\ //g' -e 's/|/,/g' | grep ,${ALANGUAGE_CODE} | tail -n 1 | cut -d ',' -f 1`"
        if [ "${ALANGUAGE_NAME2}" == "" ]
        then
            ALANGUAGE_NAME2="und"
        fi
    
        ALLAUDIO="$ALLAUDIO --language 0:${ALANGUAGE_NAME2} ${OUTPUT}/${TITLE}.${ATRACK}"
        echo $ATRACK $AINDEX $ALANGUAGE_CODE $ALANGUAGE_NAME1 $ALANGUAGE_NAME2
    done
    
    echo $ALLAUDIO
    
    if [ "$SUBTITLE_STREAMS_NUMBER_DECIMAL" != "0" ]
    then
        SUBTITLES=${OUTPUT}/${TITLE}.idx
    else
        SUBTITLES=""
    fi
    
    echo "Merging it all..."
    RUNARGS="-o ${OUTPUT}/${TITLE}.x264.mkv --default-duration 0:${DEFD} ${OUTPUT}/${TITLE}.x264 ${ALLAUDIO} ${SUBTITLES} --chapters ${OUTPUT}/${TITLE}.chapters"
    echo "mkvmerge $RUNARGS"
    ionice -c 3 nice -n 19 mkvmerge $RUNARGS 1>&2
    
    if !(test -s ${OUTPUT}/${TITLE}.x264)
    then
        echo "deleting stuff without video"
        rm -f ${OUTPUT}/${TITLE}.*
    fi
    
    mv ${OUTPUT}/${TITLE}.x264.mkv ${FINAL_DESTINATION}/${TITLE}.mkv

    if [ -a ${OUTPUT}/${TITLE}.x264.mkv ]
    then
        echo "error moving output file ${OUTPUT}/${TITLE}.x264.mkv to final destination, exiting"
        exit 1
    fi

    rm -f ${OUTPUT}/${TITLE}.*
done

echo "Done."

